
library(tidyr)
library(dplyr)
library(lme4)
data <- read.table("C:/Users/Desktop/Genova data.txt", header = FALSE, sep = "", fill = TRUE)
colnames(data) <- c(paste0("I1_R", 1:4), paste0("I2_R", 1:4), paste0("I3_R", 1:4))
data$Person <- factor(1:nrow(data))
long_data <- data %>%
  pivot_longer(
    cols = -Person,
    names_to = c("Item", "Rater"),
    names_pattern = "I(\\d)_R(\\d)",
    values_to = "Score"
  ) %>%
  mutate(
    Item = factor(paste0("I", Item)),
    Rater = factor(Rater),
    Score = suppressWarnings(as.numeric(Score))
  ) %>%
  tidyr::drop_na(Score) %>%
  mutate(
    RaterInItem = interaction(Item, Rater, sep = "_R") 
  )
## Fit G-study model ##
g_model <- lmer(Score ~ 1 +
                  (1 | Person) +
                  (1 | Item) +
                  (1 | RaterInItem) +
                  (1 | Person:Item),
                data = long_data, REML = TRUE)

summary(g_model)
## Get variance Components ##
vc <- as.data.frame(VarCorr(g_model))
print(vc)

# Pull variances
var_p   <- vc$vcov[vc$grp == "Person"]
var_i   <- vc$vcov[vc$grp == "Item"]
var_r_i  <- vc$vcov[vc$grp == "RaterInItem"]
var_pi  <- vc$vcov[vc$grp == "Person:Item"]
var_pri <- attr(VarCorr(g_model), "sc")^2
c(var_p, var_i, var_r_i, var_pi, var_pri)
# Function to compute D-study results for given items and raters
compute_G_Phi <- function(n_i, n_r) {
  rel_error <- (var_pi / n_i) + (var_pri / (n_i * n_r))
  abs_error <- (var_i / n_i) + (var_r_i / (n_i * n_r)) + (var_pi / n_i) + (var_pri / (n_i * n_r))
  
  G_coef <- var_p / (var_p + rel_error)
  Phi_coef <- var_p / (var_p + abs_error)
  
  return(data.frame(
    Relative_Error = rel_error,
    Absolute_Error = abs_error,
    G_Coefficient = G_coef,
    Phi_Coefficient = Phi_coef
  ))
}
# Apply D-study conditions
dstudy_conditions <- data.frame(
  Items = c(1, 2, 3, 4, 5, 6),
  Raters = c(12, 6, 4, 3, 2, 2)
)

# Compute for each row
d_results <- do.call(rbind, apply(dstudy_conditions, 1, function(row) {
  compute_G_Phi(as.numeric(row["Items"]), as.numeric(row["Raters"]))
}))
# Combine with the original D-study table
final_results <- cbind(dstudy_conditions, round(d_results, 4))
print(final_results)
